/* UART_ETH Client Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/queue.h"
#include "freertos/task.h"

#include "esp_system.h"
#include "esp_log.h"
#include "esp_event.h"
#include "esp_netif.h"
#include "esp_netif_net_stack.h"
#include "lwip/sockets.h"
#include "dhcpserver/dhcpserver.h"

#include "uarteth_modem.h"

#define UART_ETH_RX_TASK_PRIORITY   10
#define UART_ETH_RX_TASK_STACK_SIZE (8 * 1024)

static const char *TAG = "esp-uarteth_modem";


// UART container object
typedef struct {
    // UART device number for SIO use
    uart_port_t uart_dev;

    // UART baud rate for configuration
    uint32_t uart_baud;

    // UART TX pin for configuration
    int uart_tx_pin;

    // UART RX pin for configuration
    int uart_rx_pin;

    // QueueHandle for uart driver
    QueueHandle_t uart_queue;

    // TaskHandle for receive task
    TaskHandle_t uart_rx_task;
} esp_uarteth_uart_t;


// Modem object, implements glue logic for uarteth_driver and esp_netif
struct esp_uarteth_modem {
    // ESP base netif driver
    esp_netif_driver_base_t base;

    // Uart for use with uarteth
    esp_uarteth_uart_t uart;

    // Buffer for incoming messages
    uint8_t *buffer;
    uint32_t buffer_len;

    // Running flag
    bool running;
};


// Forward function declaration
static void esp_uarteth_modem_uart_rx_task(void *arg);
static esp_err_t esp_uarteth_modem_post_attach(esp_netif_t *esp_netif, void *args);

// Create a new uarteth netif
void *esp_uarteth_modem_create(esp_netif_t *uarteth_netif, esp_uarteth_modem_config_t *modem_config)
{
    ESP_LOGI(TAG, "%s: Creating uarteth modem (netif: %p)", __func__, uarteth_netif);

    ESP_LOGD(TAG, "%s (netif: %p)", __func__, uarteth_netif);

    esp_uarteth_modem_t *uarteth_modem = calloc(1, sizeof(esp_uarteth_modem_t));
    if (!uarteth_modem) {
        ESP_LOGE(TAG, "create netif glue failed");
        return NULL;
    }

    // Attach driver and post_attach callbacks
    uarteth_modem->base.post_attach = esp_uarteth_modem_post_attach;

    // Attach config
    uarteth_modem->buffer_len = modem_config->rx_buffer_len;

    uarteth_modem->uart.uart_dev = modem_config->uart_dev;
    uarteth_modem->uart.uart_baud = modem_config->uart_baud;
    uarteth_modem->uart.uart_rx_pin = modem_config->uart_rx_pin;
    uarteth_modem->uart.uart_tx_pin = modem_config->uart_tx_pin;

    // Return new modem, with a cast to the first item
    return &uarteth_modem->base;
}

static esp_err_t set_dhcps_dns(esp_netif_t *netif, uint32_t addr)
{
    // esp_netif_dns_info_t dns;
    // dns.ip.u_addr.ip4.addr = addr;
    // dns.ip.type = IPADDR_TYPE_V4;
    // dhcps_offer_t dhcps_dns_value = OFFER_DNS;
    ESP_ERROR_CHECK_WITHOUT_ABORT(esp_netif_dhcps_stop(netif));
    // ESP_ERROR_CHECK(esp_netif_dhcps_option(netif, ESP_NETIF_OP_SET, ESP_NETIF_DOMAIN_NAME_SERVER, &dhcps_dns_value, sizeof(dhcps_dns_value)));
    // ESP_ERROR_CHECK(esp_netif_set_dns_info(netif, ESP_NETIF_DNS_MAIN, &dns));
    ESP_ERROR_CHECK_WITHOUT_ABORT(esp_netif_dhcps_start(netif));
    return ESP_OK;
}

// Internal handler called on driver start
static esp_err_t esp_uarteth_driver_start(esp_uarteth_modem_t *uarteth_modem)
{
    ESP_LOGD(TAG, "%s: Starting UART_ETH modem (modem %p)", __func__, uarteth_modem);

    // Allocate RX buffer if one does not exist
    if (uarteth_modem->buffer == NULL) {
        uarteth_modem->buffer = malloc(uarteth_modem->buffer_len);
    }
    if (uarteth_modem->buffer == NULL) {
        ESP_LOGE(TAG, "error allocating rx buffer");
        return ESP_ERR_NO_MEM;
    }

    // Build configuration
    uart_config_t uart_config = {
        .baud_rate = uarteth_modem->uart.uart_baud,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
    };

    // Initialise uart
    ESP_ERROR_CHECK(uart_param_config(uarteth_modem->uart.uart_dev, &uart_config));

    // Set UART pins
    ESP_ERROR_CHECK(uart_set_pin(uarteth_modem->uart.uart_dev, uarteth_modem->uart.uart_tx_pin, uarteth_modem->uart.uart_rx_pin, 0, 0));

    // Install UART driver
    ESP_ERROR_CHECK(uart_driver_install(uarteth_modem->uart.uart_dev, uarteth_modem->buffer_len, uarteth_modem->buffer_len, 10, &uarteth_modem->uart.uart_queue, 0));

    // Start uarteth RX task
    uarteth_modem->running = true;
    xTaskCreate(esp_uarteth_modem_uart_rx_task, "uarteth_modem_uart_rx_task", UART_ETH_RX_TASK_STACK_SIZE, uarteth_modem, UART_ETH_RX_TASK_PRIORITY, &uarteth_modem->uart.uart_rx_task);

    // Finally, initialise uarteth network interface
    esp_netif_action_start(uarteth_modem->base.netif, 0, 0, 0);
    // esp_netif_action_connected(uarteth_modem->base.netif, 0, 0, 0);

    // Set the netif up
    struct netif *netif = esp_netif_get_netif_impl(uarteth_modem->base.netif);
    netif_set_up(netif);
    netif_set_link_up(netif);

    set_dhcps_dns(uarteth_modem->base.netif, ESP_IP4TOADDR( 10, 0, 0, 1));

    return ESP_OK;
}


esp_err_t esp_uarteth_modem_destroy(esp_uarteth_modem_t *uarteth_modem)
{
    // Stop uarteth driver
    esp_netif_action_stop(uarteth_modem->base.netif, 0, 0, 0);

    // Stop uart rx task
    vTaskDelete(uarteth_modem->uart.uart_rx_task);

    // Delete driver
    uart_driver_delete(uarteth_modem->uart.uart_dev);

    // Free uarteth interface
    free(uarteth_modem);

    return ESP_OK;
}

// Modem transmit for glue logic
static esp_err_t esp_uarteth_modem_transmit(void *uarteth_driver, void *buffer, size_t len)
{
    ESP_LOGD(TAG, "%s", __func__);
    ESP_LOG_BUFFER_HEXDUMP(TAG, buffer, (int)len, ESP_LOG_DEBUG);
    esp_uarteth_modem_t *uarteth_modem = (esp_uarteth_modem_t *) uarteth_driver;

    ESP_LOGI(TAG, "uart tx ==>: len=%d", (int)len);
    int32_t res = uart_write_bytes(uarteth_modem->uart.uart_dev, (char *)buffer, len);
    if (res < 0) {
        // Handle errors
        ESP_LOGE(TAG, "%s: uart_write_bytes error %d", __func__, res);
        return ESP_FAIL;
    }
    return ESP_OK;
}

static void uarteth_free_rx_buffer(void *h, void* buffer)
{
    free(buffer);
}

extern const uint8_t g_uarteth_mac[6];
// Post-attach handler for netif
static esp_err_t esp_uarteth_modem_post_attach(esp_netif_t *esp_netif, void *args)
{
    esp_uarteth_modem_t *uarteth_modem = (esp_uarteth_modem_t *)args;

    ESP_LOGD(TAG, "%s (netif: %p args: %p)", __func__, esp_netif, args);

    const esp_netif_driver_ifconfig_t driver_ifconfig = {
        .driver_free_rx_buffer = uarteth_free_rx_buffer,
        .transmit = esp_uarteth_modem_transmit,
        .handle = uarteth_modem,
    };

    uarteth_modem->base.netif = esp_netif;
    ESP_ERROR_CHECK(esp_netif_set_driver_config(esp_netif, &driver_ifconfig));

    struct netif *netif = esp_netif_get_netif_impl(esp_netif);
    memcpy(netif->hwaddr, g_uarteth_mac, 6);

    esp_uarteth_driver_start(uarteth_modem);

    return ESP_OK;
}

#if 1
static void esp_uarteth_modem_uart_rx_task(void *arg)
{
    esp_uarteth_modem_t *uarteth_modem = (esp_uarteth_modem_t *) arg;

    ESP_LOGD(TAG, "Start UART_ETH modem RX task (uarteth_modem %p)", uarteth_modem);
    ESP_LOGD(TAG, "Uart: %d, buffer: %p (%d bytes)", uarteth_modem->uart.uart_dev, uarteth_modem->buffer, uarteth_modem->buffer_len);

    while (uarteth_modem->running == true) {
        // Read data from the UART
        int len = uart_read_bytes(uarteth_modem->uart.uart_dev, uarteth_modem->buffer, uarteth_modem->buffer_len, 1 / portTICK_PERIOD_MS);
        if (len > 0) {
            uint8_t *buffer = malloc(len);
            if (buffer == NULL) {
                ESP_LOGE(TAG, "uart rcv buffer malloc failed, len=%d", len);
                vTaskDelay(100);
                continue;
            }
            memcpy(buffer, uarteth_modem->buffer, len);

            // Log uarteth RX data
            ESP_LOGI(TAG, "uart rx: len=%d", len);
            ESP_LOG_BUFFER_HEX_LEVEL(TAG, buffer, len, ESP_LOG_DEBUG);

            // Pass received bytes in to uarteth interface
            esp_netif_receive(uarteth_modem->base.netif, buffer, len, NULL);
        }

        // Yeild to allow other tasks to progress
        vTaskDelay(1 * portTICK_PERIOD_MS);
    }
}
#else
static void esp_uarteth_modem_uart_rx_task(void *arg)
{
    esp_uarteth_modem_t *uarteth_modem = (esp_uarteth_modem_t *)arg;
    uart_event_t event;
    size_t buffered_size;

    for(;;) {
        //Waiting for UART event.
        if(xQueueReceive(uarteth_modem->uart.uart_queue, (void * )&event, (portTickType)portMAX_DELAY)) {
            switch(event.type) {
                //Event of UART receving data
                /*We'd better handler data event fast, there would be much more data events than
                other types of events. If we take too much time on data event, the queue might
                be full.*/
                case UART_DATA: {
                    ESP_LOGI(TAG, "[UART DATA]: %d", event.size);
                    int len = uart_read_bytes(uarteth_modem->uart.uart_dev, uarteth_modem->buffer, event.size, portMAX_DELAY);
                    if (len > 0) {
                        uint8_t *buffer = malloc(len);
                        if (buffer == NULL) {
                            ESP_LOGE(TAG, "uart rcv buffer malloc failed, len=%d", len);
                            vTaskDelay(100);
                            continue;
                        }
                        memcpy(buffer, uarteth_modem->buffer, len);

                        // Log uarteth RX data
                        ESP_LOGI(TAG, "uart rx: len=%d", len);
                        ESP_LOG_BUFFER_HEX_LEVEL(TAG, buffer, len, ESP_LOG_DEBUG);

                        // Pass received bytes in to uarteth interface
                        esp_netif_receive(uarteth_modem->base.netif, buffer, len, NULL);
                    }
                    break;
                }

                //Event of HW FIFO overflow detected
                case UART_FIFO_OVF:
                    ESP_LOGI(TAG, "hw fifo overflow");
                    // If fifo overflow happened, you should consider adding flow control for your application.
                    // The ISR has already reset the rx FIFO,
                    // As an example, we directly flush the rx buffer here in order to read more data.
                    uart_flush_input(uarteth_modem->uart.uart_dev);
                    xQueueReset(uarteth_modem->uart.uart_queue);
                    break;
                //Event of UART ring buffer full
                case UART_BUFFER_FULL:
                    ESP_LOGI(TAG, "ring buffer full");
                    // If buffer full happened, you should consider encreasing your buffer size
                    // As an example, we directly flush the rx buffer here in order to read more data.
                    uart_flush_input(uarteth_modem->uart.uart_dev);
                    xQueueReset(uarteth_modem->uart.uart_queue);
                    break;
                //Event of UART RX break detected
                case UART_BREAK:
                    ESP_LOGI(TAG, "uart rx break");
                    break;
                //Event of UART parity check error
                case UART_PARITY_ERR:
                    ESP_LOGI(TAG, "uart parity error");
                    break;
                //Event of UART frame error
                case UART_FRAME_ERR:
                    ESP_LOGI(TAG, "uart frame error");
                    break;
                //UART_PATTERN_DET
                case UART_PATTERN_DET:
                    ESP_LOGI(TAG, "UART_PATTERN_DET");
                    break;
                //Others
                default:
                    ESP_LOGI(TAG, "uart event type: %d", event.type);
                    break;
            }
        }
    }
    vTaskDelete(NULL);
}
#endif


