/* UART_ETH Client Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/
#include <string.h>
#include <stdio.h>
#include "driver/uart.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"

#include "esp_system.h"
#include "esp_log.h"
#include "esp_event.h"
#include "esp_netif.h"

#include "esp_eth.h"
#include "nvs_flash.h"

#include "uarteth_modem.h"

static const char *TAG = "uarteth";

#define CONFIG_EXAMPLE_UART_TX_PIN 4
#define CONFIG_EXAMPLE_UART_RX_PIN 16
#define CONFIG_EXAMPLE_UART_BAUD 4000000

#define CONFIG_EXAMPLE_WIFI_SSID "eth2wifi_ssid"
#define CONFIG_EXAMPLE_WIFI_PASSWORD "12345678"
#define CONFIG_EXAMPLE_MAX_STA_CONN 4
#define CONFIG_EXAMPLE_WIFI_CHANNEL 1

static esp_eth_handle_t s_eth_handle = NULL;
static bool s_sta_is_connected = false;
static bool s_ethernet_is_connected = false;

static const esp_netif_ip_info_t s_uarteth_ip4 = {
        .ip = { .addr = ESP_IP4TOADDR( 10, 0, 0, 1) },
        .netmask = {.addr = ESP_IP4TOADDR( 255, 0, 0, 0),}
};
const uint8_t g_uarteth_mac[6] = {0x98, 0x43, 0xfa, 0x04, 0xcd, 0xf1};

// Initialise the UART_ETH interface
esp_netif_t *uarteth_if_init(void)
{
    ESP_LOGI(TAG, "Initialising UART-ETH interface");

    esp_netif_inherent_config_t base_cfg = ESP_NETIF_INHERENT_DEFAULT_ETH();

    base_cfg.flags =  (esp_netif_flags_t)(ESP_NETIF_DHCP_SERVER | ESP_NETIF_FLAG_GARP | ESP_NETIF_FLAG_EVENT_IP_MODIFIED);
    base_cfg.get_ip_event = 0;
    base_cfg.lost_ip_event = 0;
    base_cfg.ip_info = &s_uarteth_ip4;
    memcpy(base_cfg.mac, g_uarteth_mac, 6);

    esp_netif_config_t cfg = {          .base = &base_cfg,
                                        .driver = NULL,
                                        .stack = ESP_NETIF_NETSTACK_DEFAULT_ETH };

    esp_netif_t *uarteth_netif = esp_netif_new(&cfg);

    ESP_LOGI(TAG, "Initialising UART_ETH modem");

    esp_uarteth_modem_config_t modem_cfg = {
        .uart_dev = UART_NUM_1,

        .uart_tx_pin = CONFIG_EXAMPLE_UART_TX_PIN,
        .uart_rx_pin = CONFIG_EXAMPLE_UART_RX_PIN,
        .uart_baud = CONFIG_EXAMPLE_UART_BAUD,
        .rx_buffer_len = 4096,
    };

    void *uarteth_modem = esp_uarteth_modem_create(uarteth_netif, &modem_cfg);
    ESP_ERROR_CHECK(esp_netif_attach(uarteth_netif, uarteth_modem));

    ESP_LOGI(TAG, "UART_ETH init complete");

    return uarteth_netif;
}

void app_main(void)
{
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    // Setup networking
    esp_netif_init();

    esp_log_level_set("*", ESP_LOG_DEBUG);

    // Create event loop
    ESP_ERROR_CHECK(esp_event_loop_create_default());

    // Setup uarteth interface
    uarteth_if_init();

    while (true)
    {
        vTaskDelay(100);
    }
    
}
