/* UART_ETH Client Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"

#include "esp_system.h"
#include "esp_log.h"
#include "esp_event.h"
#include "esp_netif.h"

#include "esp_wifi.h"
#include "nvs_flash.h"
#include "esp_private/wifi.h"
#include "driver/uart.h"
#include "lwip/prot/ethernet.h"
#include "lwip/prot/ip4.h"
#include "lwip/prot/udp.h"
#include "lwip/prot/tcp.h"

static const char *TAG = "ap2uart";

#define CONFIG_UART_DEV UART_NUM_1
#define CONFIG_EXAMPLE_UART_TX_PIN 4
#define CONFIG_EXAMPLE_UART_RX_PIN 16
#define CONFIG_EXAMPLE_UART_BAUD 4000000
#define CONFIG_UART_DRV_BUFFER_SIZE 2048
#define CONFIG_UART_RX_TASK_PRIORITY 10
#define CONFIG_UART_RX_TASK_STACK_SIZE (6 * 1024)

#define CONFIG_EXAMPLE_WIFI_SSID "ap2uart_ssid"
#define CONFIG_EXAMPLE_WIFI_PASSWORD "12345678"
#define CONFIG_EXAMPLE_MAX_STA_CONN 4
#define CONFIG_EXAMPLE_WIFI_CHANNEL 1

static bool g_sta_is_connected = false;
static bool g_uart_is_connected = false;

static const uint8_t g_uarteth_mac[6] = {0x98, 0x43, 0xfa, 0x04, 0xcd, 0xf1};

static void uart_rx_task(void *arg)
{
    ESP_LOGI(TAG, "Start uart RX task");
    uint8_t rx_budffer[2048];
    int res;

    g_uart_is_connected = true;
    while (true) {
        // Read data from the UART
        int len = uart_read_bytes(CONFIG_UART_DEV, rx_budffer, 2048, 1 / portTICK_PERIOD_MS);
        if (len > 0) {
            // Log uarteth RX data
            ESP_LOGI(TAG, "uart rx <==: len=%d", len);
            // ESP_LOG_BUFFER_HEX_LEVEL(TAG, rx_budffer, len, ESP_LOG_DEBUG);

            // Pass received bytes in to ap interface
            if (g_sta_is_connected) {
                res = esp_wifi_internal_tx(WIFI_IF_AP, rx_budffer, len);
                if (res != ESP_OK) {
                    ESP_LOGE(TAG, "wifi_internal_tx failed, RET=%d", res);
                }
            }
        }

        // Yeild to allow other tasks to progress
        vTaskDelay(1 * portTICK_PERIOD_MS);
    }
}

// Internal handler called on driver start
static esp_err_t initialize_uart()
{
    ESP_LOGI(TAG, "%s: dev=%d baud=%d", __func__, CONFIG_UART_DEV, CONFIG_EXAMPLE_UART_BAUD);

    // Build configuration
    uart_config_t uart_config = {
        .baud_rate = CONFIG_EXAMPLE_UART_BAUD,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
    };

    // Initialise uart
    ESP_ERROR_CHECK(uart_param_config(CONFIG_UART_DEV, &uart_config));

    // Set UART pins
    ESP_ERROR_CHECK(uart_set_pin(CONFIG_UART_DEV, CONFIG_EXAMPLE_UART_TX_PIN, CONFIG_EXAMPLE_UART_RX_PIN, 0, 0));

    // Install UART driver
    ESP_ERROR_CHECK(uart_driver_install(CONFIG_UART_DEV, CONFIG_UART_DRV_BUFFER_SIZE, CONFIG_UART_DRV_BUFFER_SIZE, 0, NULL, 0));

    // Start uarteth RX task
    xTaskCreate(uart_rx_task, "uart_rx_task", CONFIG_UART_RX_TASK_STACK_SIZE, NULL, CONFIG_UART_RX_TASK_PRIORITY, NULL);

    return ESP_OK;
}

#define ETH_HDR_LEN 14
#define IPV4_HDR_MIN_LEN 20
#define ETH_FRM_TYPE_IPV6 0x86dd
#define ETH_FRM_TYPE_IPV4 0x0800

#define IP_PROTO_ICMP 1
#define IP_PROTO_TCP 6
#define IP_PROTO_UDP 17

#define UDP_HDR_LEN 8
#define UDP_DNS_PORT 53
#define UDP_MDNS_PORT 5353
#define UDP_SSDP_PORT 1900
#define UDP_NBNS_PORT 137
#define UDP_LLMNR_PORT 5355
#define UDP_WSD_PORT 3702  // WS-Discovery

#define TCP_HDR_MIN_LEN 20
#define TCP_DNS_PORT 53
/* 报文过滤，滤除不关心的报文然后丢弃
 * 返回值 true:报文被过滤掉 false:报文不被过滤
 */
static bool pkt_wifi_filter(void *buffer, uint16_t len)
{
    if (buffer == NULL || len <= ETH_HDR_LEN) {
        return true;
    }
    uint8_t *frame = (uint8_t *)buffer;

    /* Level1: 链路层过滤 |dst_mac(6B)|src_mac(6B)|type(2B)| */
    uint16_t type = (frame[12] << 8) | frame[13];
    if (type == ETH_FRM_TYPE_IPV6) {
        return true; /* 过滤掉IPV6 */
    }

    /* Level2: IP层过滤 */
    if (type != ETH_FRM_TYPE_IPV4) {
        return false; /* 非IPV4的其他类型, 不过滤 */
    }
    if (len <= (ETH_HDR_LEN + IPV4_HDR_MIN_LEN)) {
        return true;
    }

    struct ip_hdr *ip_hdr = (struct ip_hdr *)&frame[ETH_HDR_LEN];
     /* 仅接收icmp(1)、tcp(6)、udp(17), 其它过滤掉 */
    if (ip_hdr->_proto != IP_PROTO_ICMP && ip_hdr->_proto != IP_PROTO_TCP && ip_hdr->_proto != IP_PROTO_UDP) {
        return true;
    }

    /* Level3: 传输层过滤 */
    /* UDP报文过滤 */
    if (ip_hdr->_proto == IP_PROTO_UDP) {
        uint16_t min_len = ETH_HDR_LEN + IPH_HL_BYTES(ip_hdr) + UDP_HDR_LEN;
        if (len <= min_len) {
            return true;
        }
        struct udp_hdr *udp_hdr = (struct udp_hdr *)&frame[ETH_HDR_LEN + IPH_HL_BYTES(ip_hdr)];
        uint16_t dst_port = ntohs(udp_hdr->dest);
        if (dst_port == UDP_DNS_PORT || dst_port == UDP_MDNS_PORT || dst_port == UDP_SSDP_PORT ||
            dst_port == UDP_NBNS_PORT || dst_port == UDP_LLMNR_PORT || dst_port == UDP_WSD_PORT) {
            return true;
        }
    }

     /* TCP报文过滤 */
    if (ip_hdr->_proto == IP_PROTO_TCP) {
        uint16_t min_len = ETH_HDR_LEN + IPH_HL_BYTES(ip_hdr) + TCP_HDR_MIN_LEN;
        if (len <= min_len) {
            return true;
        }
        struct tcp_hdr *tcp_hdr = (struct tcp_hdr *)&frame[ETH_HDR_LEN + IPH_HL_BYTES(ip_hdr)];
        uint16_t dst_port = ntohs(tcp_hdr->dest);
        if (dst_port == TCP_DNS_PORT) {
            return true;
        }
    }

    return false;
}

// Forward packets from Wi-Fi to uart
static esp_err_t pkt_wifi2uart(void *buffer, uint16_t len, void *eb)
{
    if (pkt_wifi_filter(buffer, len) == true) {
        esp_wifi_internal_free_rx_buffer(eb);
        return ESP_OK;
    }

    ESP_LOGI(TAG, "send to uart: len=%d", len);
    // ESP_LOG_BUFFER_HEX_LEVEL(TAG, buffer, len, ESP_LOG_INFO);
    if (g_uart_is_connected) {
        int res = uart_write_bytes(CONFIG_UART_DEV, buffer, len);
        if (res != len) {
            ESP_LOGE(TAG, "UART send packet failed, len=%d res=%d", len, res);
        }
    }
    esp_wifi_internal_free_rx_buffer(eb);
    return ESP_OK;
}

// Event handler for Wi-Fi
static void wifi_event_handler(void *arg, esp_event_base_t event_base,
                               int32_t event_id, void *event_data)
{
    static uint8_t s_con_cnt = 0;
    switch (event_id) {
    case WIFI_EVENT_AP_STACONNECTED:
        ESP_LOGI(TAG, "Wi-Fi AP got a station connected");
        if (!s_con_cnt) {
            g_sta_is_connected = true;
            esp_wifi_internal_reg_rxcb(WIFI_IF_AP, pkt_wifi2uart);
        }
        s_con_cnt++;
        break;
    case WIFI_EVENT_AP_STADISCONNECTED:
        ESP_LOGI(TAG, "Wi-Fi AP got a station disconnected");
        s_con_cnt--;
        if (!s_con_cnt) {
            g_sta_is_connected = false;
            esp_wifi_internal_reg_rxcb(WIFI_IF_AP, NULL);
        }
        break;
    default:
        break;
    }
}

static void initialize_wifi(void)
{
    ESP_ERROR_CHECK(esp_event_handler_register(WIFI_EVENT, ESP_EVENT_ANY_ID, wifi_event_handler, NULL));
    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_wifi_init(&cfg));
    ESP_ERROR_CHECK(esp_wifi_set_storage(WIFI_STORAGE_RAM));
    wifi_config_t wifi_config = {
        .ap = {
            .ssid = CONFIG_EXAMPLE_WIFI_SSID,
            .ssid_len = strlen(CONFIG_EXAMPLE_WIFI_SSID),
            .password = CONFIG_EXAMPLE_WIFI_PASSWORD,
            .max_connection = CONFIG_EXAMPLE_MAX_STA_CONN,
            .authmode = WIFI_AUTH_WPA_WPA2_PSK,
            .channel = CONFIG_EXAMPLE_WIFI_CHANNEL // default: channel 1
        },
    };
    if (strlen(CONFIG_EXAMPLE_WIFI_PASSWORD) == 0) {
        wifi_config.ap.authmode = WIFI_AUTH_OPEN;
    }
    ESP_ERROR_CHECK(esp_wifi_set_mode(WIFI_MODE_AP));
    ESP_ERROR_CHECK(esp_wifi_set_config(WIFI_IF_AP, &wifi_config));

    esp_wifi_set_mac(WIFI_IF_AP, g_uarteth_mac);
    ESP_ERROR_CHECK(esp_wifi_start());
}

void app_main(void)
{
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    // Setup networking
    esp_netif_init();

    esp_log_level_set("*", ESP_LOG_DEBUG);

    // Create event loop
    ESP_ERROR_CHECK(esp_event_loop_create_default());

    initialize_uart();

    initialize_wifi();

    while (true)
    {
        vTaskDelay(100);
    }    
}
